package com.accelops.phoenix.servicenow;

import java.io.IOException;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import org.apache.http.HttpEntity;
import org.apache.http.ParseException;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.AbstractHttpEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;

import com.github.rholder.retry.RetryException;
import com.github.rholder.retry.Retryer;
import com.github.rholder.retry.RetryerBuilder;
import com.github.rholder.retry.StopStrategies;
import com.github.rholder.retry.WaitStrategies;
import com.ph.phoenix.http.HttpClientUtil;

public class ServiceNowSoapClient {

    static final Logger logger = Logger.getLogger(ServiceNowSoapClient.class.getName());

    private UsernamePasswordCredentials credential;

    protected String _soapHost = "";
    protected Integer _soapPort = 80;

    protected String soapResultMessage = "";
    protected Integer soapStatusCode = 0;

    protected ContentType _contentType;

    private HttpEntity _httpEntity;

    private String _soapAction;

    private String _soapMethodURI;
    
    private Retryer<Boolean> retrier;

    
    
    public ServiceNowSoapClient(String host, Integer port, String user, String password) {
        _soapHost = host;
        _soapPort = port;
        credential = new UsernamePasswordCredentials(user, password);
        
        createRetrier();
        
    }

    public ServiceNowSoapClient(String host, Integer port, String user, String password, ContentType contentType) {
        _soapHost = host;
        _soapPort = port;
        credential = new UsernamePasswordCredentials(user, password);
        this._contentType = contentType;
        
        createRetrier();
    }

    private void createRetrier(){
        retrier = RetryerBuilder.<Boolean>newBuilder()
                .retryIfExceptionOfType(IOException.class)
                .withStopStrategy(StopStrategies.stopAfterAttempt(3))
                .withWaitStrategy(WaitStrategies.fixedWait(3, TimeUnit.SECONDS))
                .build();
    }
    
    
    public Integer getSoapStatusCode() {
        return soapStatusCode;
    }

    public String getSoapResultMessage() {
        return soapResultMessage;
    }

    public int executeSoapMethod(String soapMethodURI, String soapAction, HttpEntity httpEntity) throws ClientProtocolException, ParseException, IOException {

        this._soapMethodURI = soapMethodURI;
        this._soapAction = soapAction;

        if (httpEntity instanceof AbstractHttpEntity) {
            return sendWithContentType((AbstractHttpEntity) httpEntity);
        } else {
            this._httpEntity = httpEntity;
        }

        return send();
    }

    private int sendWithContentType(AbstractHttpEntity soapMessageEntity) throws ClientProtocolException, ParseException, IOException {
        if (_contentType != null) {
            soapMessageEntity.setContentType(_contentType.toString());
        } else {
            soapMessageEntity.setContentType(ServiceNowIntegration.Default_ContentType.toString());
        }

        _httpEntity = soapMessageEntity;

        return send();
    }

    private int send() throws IOException {

        StringBuilder soapUrl = new StringBuilder("https://");

        if (!_soapMethodURI.startsWith("/")) {
            _soapMethodURI = "/" + _soapMethodURI;
        }

        soapUrl.append(_soapHost).append(":").append(_soapPort).append(_soapMethodURI);

        HttpPost httpPost = new HttpPost(soapUrl.toString());

        httpPost.setHeader("SOAPAction", _soapAction);

        httpPost.setEntity(_httpEntity);

        try
        {
            retrier.call(new Callable<Boolean>() {
                
                
                @Override
                    public Boolean call() throws Exception {
                    
                        CloseableHttpClient httpclient = null;
                        
                        CloseableHttpResponse httpResponse = null;
                        
                        try  {

                            httpclient  = HttpClientUtil.getHttpClient(credential.getUserName(), credential.getPassword());
                            httpResponse = httpclient.execute(httpPost);

                            HttpEntity respHttpEntity = httpResponse.getEntity();

                            soapStatusCode = httpResponse.getStatusLine().getStatusCode();
                            soapResultMessage = EntityUtils.toString(respHttpEntity);
                            
                        } finally {
                            if (httpclient != null) {
                                httpclient.close();
                            }
                            if (httpResponse != null) {
                                httpResponse.close();
                            }
                        }
        
                        return true;
                    }
                }
            );
        }
        catch(RetryException ex){
            logger.info("Failed on ServiceNow integration after 3 times due to "+ ex.getCause().getMessage());
        }
        catch(ExecutionException ex){
            throw new RuntimeException(ex.getCause());
        }
       
        return soapStatusCode;
    }

}
