package com.accelops.phoenix.servicenow;

import java.io.IOException;
import java.io.StringReader;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Level;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathFactory;

import org.apache.commons.lang.StringUtils;
import org.apache.http.entity.StringEntity;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import com.accelops.service.ExternalObjectInfo;
import com.accelops.service.ObjectType;
import com.accelops.service.ServiceContext;
import com.accelops.service.exception.SOAPAuthorizationException;
import com.accelops.service.exception.ServiceNowIntegrationException;

public class IncidentServiceNowIntegration extends
		AbstractServiceNowIntegration {

    Map<String,String> unLinkedDevices; 
    
 	protected IncidentServiceNowIntegration(ServiceContext context) {
		super(context);
	}

	
	/**
	 * ServiceNow has bug when insert/get multiple recorders, so that we do not support to insert multiple incidents at one time.
	 * 
	 * 
	 *  
	 */
	@Override
	protected void execute() throws Exception
	{
	    
		// create the source document based on source xml
        //
        Document document = XMLUtils.createDocument(this.getIntegratedXML(), false);
        
        // process incidents
        
        NodeList incidentList = document.getElementsByTagName("incident");
        
        if (incidentList!=null && incidentList.getLength()> 0)
        {
	        int incidentLength = incidentList.getLength();
	        
	        
	        
	        
	        for (int i=0; i < incidentLength; i++) 
	        {
	            
	            // create incident document
	            //
	            Node incident = (Node)incidentList.item(i);
	            
	            try
	            {
	                
	                ExternalObjectInfo incidentInfo = processIncident(XMLUtils.nodeToString(incident),  getXslPath());
                    
                    if (incidentInfo != null)
                    {
                        externalObjList.add(incidentInfo);
                    }
	            }
	            catch(SOAPAuthorizationException |  ServiceNowIntegrationException ex)
                {
	                throw ex;
                }
                catch(Exception ex)
                {
                    logger.info(ex.getMessage());
                }
	        }
	        
	        
	        
	        svcContext.setAttribute(ServiceContext.LINK_DEVICES, unLinkedDevices);
        }
	}
	
	/**
     * 
     * @param incidentXml
     * @param xslPath
     * @return
     * @throws ParserConfigurationException
     * @throws SAXException
     * @throws IOException
     * @throws Exception
     */
    protected ExternalObjectInfo processIncident(String incidentXml, String xslPath) throws ParserConfigurationException, SAXException, IOException, Exception {
        
        ExternalObjectInfo incidentInfo = null;
        
        String sysId = getSnIncidentSysId(incidentXml, xslPath);
        
        if (StringUtils.isBlank(sysId))
        {
            incidentInfo = insertIncident(incidentXml, xslPath);
        }
        else
        {
            updateSnIncident(sysId, incidentXml, xslPath);
        }
        
        return incidentInfo;
    }
    
    /**
     * 
     * @param incidentXml
     * @param xslPath
     * @return
     * @throws Exception
     */
    protected ExternalObjectInfo insertIncident(String incidentXml, String xslPath) throws Exception {
        
        ExternalObjectInfo incidentInfo = null;
        
        Document document = assembleIncidentTransformDoc(incidentXml,xslPath);
        
        // transform to servicenow incident
        //
        String xslIncidentInsert = xslPath + "/ServiceNow-Incident-Insert.xsl";
        String incidentServiceNow = XMLUtils.transform(document, xslIncidentInsert);
        
       
        int statusCode = getSoapClient().executeSoapMethod(getIncidentSOAPURI(), "http://www.service-now.com/incident/insert", new StringEntity(incidentServiceNow));
        
        if (statusCode == 200) {
            
            String externalId = "";
            String internalId = "";
            
            // parse servicenow incident id
            
            externalId = this.parseInsertResponse(getSoapClient().getSoapResultMessage());
            
            // parse accelops incident id
            
            XPath xPath = XPathFactory.newInstance().newXPath();
            internalId = xPath.compile("/insert/incident/@id").evaluate(document);
            

            // create external object
            // 
            incidentInfo = new ExternalObjectInfo();
            incidentInfo.setObjectType(ObjectType.Incident);
            incidentInfo.setObjectId(externalId);
            incidentInfo.setInternalObjectId(internalId);
            incidentInfo.setOrganizationName("servicenow");
            
        }
        else 
        {
            logger.log(Level.SEVERE, "Failed to insert incident: [status = " + statusCode+"];[message="+getSoapClient().getSoapResultMessage()+"]");
            faultProcess(String.valueOf(statusCode));
        }
        return incidentInfo;
    }
    
    
    protected Document assembleIncidentTransformDoc(String incidentXml,
            String xslPath) throws Exception{
        String insertIncidentXml = "<insert>" + incidentXml + "</insert>";
        
        Document document = XMLUtils.createDocument(insertIncidentXml, false);

        // query for servicenow configuration item
        //
        String serviceNowCI = getServiceNowCI(incidentXml, xslPath);
        
        if (StringUtils.isNotBlank(serviceNowCI)) {
            Element ciElement = document.createElement("cmdb_ci");
            if (ciElement != null) {
                ciElement.setTextContent(serviceNowCI);
                document.getFirstChild().getFirstChild().appendChild(ciElement);
            }
        }
        
        Properties customProperties = loadCustomProperties(xslPath);
        if (customProperties!=null && !customProperties.isEmpty())
        {
            Node parentNode = document.getFirstChild().getFirstChild();
            for(Enumeration<?> propNames = customProperties.propertyNames();propNames.hasMoreElements();)
            {
                String propName = (String)propNames.nextElement();
                Element ciElement = document.createElement(propName);
                ciElement.setTextContent(customProperties.getProperty(propName));
                parentNode.appendChild(ciElement);
            }
        }
        
        return document;
    }


    protected Properties loadCustomProperties(String xslPath)
    {
        return null;
    }
    
    protected String getIncidentInsertAction() {
        return "http://www.service-now.com/incident/insert";
    }
    
    /**
     * 
     * @param sysId
     * @param incidentXml
     * @param xslPath
     *
     * @throws ParserConfigurationException
     * @throws SAXException
     * @throws IOException
     * @throws Exception
     */
    protected void updateSnIncident(String sysId, String incidentXml, String xslPath) throws ParserConfigurationException, SAXException, IOException, Exception {
        
        // Check external status, do nothing if it's already        closed.
        // 
        Document document = XMLUtils.createDocument(incidentXml, false);
        
        XPath xPath = XPathFactory.newInstance().newXPath();
        
        String status = xPath.compile("/incident/incidentStatus").evaluate(document);
        
        
        if (StringUtils.isNotBlank(status) && StringUtils.isNotBlank(sysId)){
            
            String xslIncidentUpdateOrResolve = xslPath + "/ServiceNow-Incident-Insert.xsl";
            
            if (StringUtils.isNumeric(status) && Integer.parseInt(status) == 0){
                
                incidentXml = "<update>" + incidentXml + "</update>";
            }
            else{
                incidentXml = "<resolve>" + incidentXml + "</resolve>";                
            }
            
            document = XMLUtils.createDocument(incidentXml, false);
            
            Element ciElement = document.createElement("sys_id");
            if (ciElement != null) {
                ciElement.setTextContent(sysId);
                document.getFirstChild().getFirstChild().appendChild(ciElement);
            }
            
            // transform to ServiceNow incident
            //
            String incidentSN = XMLUtils.transform(document, xslIncidentUpdateOrResolve);
            
            logger.info("Update Incident " + incidentSN);
            
            int statusCode = getSoapClient().executeSoapMethod("/incident.do?SOAP", "http://www.service-now.com/incident/update", new StringEntity(incidentSN));
    
            if (statusCode != 200) {
                logger.log(Level.SEVERE, "Failed to update incident: " + getSoapClient().getSoapResultMessage());
                faultProcess(String.valueOf(statusCode));
            }
        }
    }
    
    
    /**
     * 
     * @param soapResult
     * @return
     * @throws ParserConfigurationException
     * @throws SAXException
     * @throws IOException
     */
    protected String parseInsertResponse(String soapResult) throws ParserConfigurationException, SAXException, IOException {

        String incidentNumber = "";

        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();

        dbf.setNamespaceAware(false);
        DocumentBuilder docBuilder = dbf.newDocumentBuilder();
        
        InputSource inputSource = new InputSource();
        inputSource.setCharacterStream(new StringReader(soapResult));

        Document xmlDocument = docBuilder.parse(inputSource);

        Element elRoot = xmlDocument.getDocumentElement();

        NodeList listSysId = elRoot.getElementsByTagName(getIncidentFieldName());

        if ((listSysId != null) && (listSysId.getLength() > 0)) {
            Element elSysId = (Element)listSysId.item(0);
            incidentNumber = elSysId.getFirstChild().getNodeValue();
        }
        else
        {
            NodeList listStatus = elRoot.getElementsByTagName("status");
            
            if ((listStatus != null) && (listStatus.getLength() > 0)){
                
                Element statusEle = (Element)listStatus.item(0);
                
                String status = statusEle.getFirstChild().getNodeValue();
                
                if (status!=null && status.equalsIgnoreCase("error"))
                {
                    NodeList listError = elRoot.getElementsByTagName("error_message");
                    
                    if ((listError != null) && (listError.getLength() > 0))
                    {
                        Element errorEle = (Element)listError.item(0);
                        
                        String errMsg = errorEle.getFirstChild().getNodeValue();
                        if (StringUtils.isNotBlank(errMsg))
                        {
                            throw new RuntimeException(errMsg);
                        }
                    }
                }
            }
            
            NodeList listDisplayName = elRoot.getElementsByTagName("display_name");

            if ((listDisplayName != null) && (listDisplayName.getLength() > 0)) {
                
                Element elDisplayName = (Element)listDisplayName.item(0);
                
                String fieldName = elDisplayName.getFirstChild().getNodeValue();
                
                if (getIncidentFieldName().equals(fieldName))
                {
                    NodeList listValueNode = elRoot.getElementsByTagName("display_value");
                    
                    if ((listValueNode != null) && (listValueNode.getLength() > 0)) {
                        
                        Element elValue = (Element)listValueNode.item(0);
                        incidentNumber = elValue.getFirstChild().getNodeValue();
                    }
                }
            }
        }
        
        return incidentNumber;        
    }
    
    /**
     * Get sys_id in incident from service now 
     * Web service update called without a sys_id, ignored
     * 
     * @param incidentXml
     * @param xslPath
     * @return
     * @throws ParserConfigurationException
     * @throws SAXException
     * @throws IOException
     * @throws Exception
     */
    protected String getSnIncidentSysId(String incidentXml, String xslPath) throws ParserConfigurationException, SAXException, IOException, Exception {
        
            String sysId = "";

            Document document = XMLUtils.createDocument(incidentXml, false);
            
            Element elRoot = document.getDocumentElement();

            NodeList listExtId = elRoot.getElementsByTagName("externalId");

            if ((listExtId != null) && (listExtId.getLength() > 0)) 
            {
                int index = 0;
                
                String incidentExtNumber = "";
                
                String queryXsl = xslPath + "/ServiceNow-Incident-GetRecs.xsl";
                
                String incidentQuery = XMLUtils.transform(document, queryXsl);
                do
                {
                    Node externalIdNode = listExtId.item(index++);
                    
                    if(externalIdNode.getNodeType() == Node.ELEMENT_NODE)
                    {
                        
                        incidentExtNumber = externalIdNode.getTextContent();
                        if (StringUtils.isNotBlank(incidentExtNumber))
                        {
                            int statusCode = getSoapClient().executeSoapMethod("/incident.do?SOAP", "http://www.service-now.com/incident/getRecords", new StringEntity(incidentQuery));
                            if (statusCode == 200) {
                             
                                sysId = this.parseSoapResultSysId(getSoapClient().getSoapResultMessage());
                                
                                break;
                            }
                        }
                    }
                    
                }while(index < listExtId.getLength());
            }
           return sysId;
    }

    /**
     * 
     * @param incidentXml
     * @param xslPath
     * @return
     * @throws ParserConfigurationException
     * @throws SAXException
     * @throws IOException
     * @throws Exception
     */
    protected String getServiceNowCI(String incidentXml, String xslPath) throws ParserConfigurationException, SAXException, IOException, Exception {
        
        // String ciId = "e141cdfe0fd12100138af25be1050ec3";
        String ciId = "";
        
        Document document = XMLUtils.createDocument(incidentXml, false);
        
        NodeList nodes = document.getElementsByTagName("externalDeviceId");
        
        if (nodes.getLength()>0){
            Node node = nodes.item(0);
            ciId = node.getTextContent();
        }
        
        

        if (StringUtils.isBlank(ciId))
        {
            String hostName = XMLUtils.getElementValue(document.getDocumentElement(), "hostName");
            
            if (unLinkedDevices == null)
            {
                unLinkedDevices = new HashMap<>();
            }
            
            if (hostName!=null)
            {
                ciId =unLinkedDevices.get(hostName); 
            }
            
            if (StringUtils.isBlank(ciId))
            {
                // transform to servicenow incident
                
                String queryXsl = xslPath + "/ServiceNow-Incident-QueryCI.xsl";
                
                String cmdbQuery = XMLUtils.transform(document, queryXsl);
                
                
                int statusCode = getSoapClient().executeSoapMethod("/cmdb_ci.do?SOAP", "http://www.service-now.com/cmdb_ci/getRecords", new StringEntity(cmdbQuery));
                if (statusCode != 200)
                    return ciId;
                
                // parse servicenow cmdb ci sys id
                //    
                ciId = parseIncidentDeviceSysId(getSoapClient().getSoapResultMessage());
                
                if (hostName!=null && StringUtils.isNotBlank(ciId))
                {
                    unLinkedDevices.put(hostName, ciId);
                }
            }
        }
        return ciId;
    }
    
    /**
     * 
     * @param soapResult
     * @return sys_id -- id from service now
     * 
     * @throws SAXException
     * @throws IOException
     * @throws ParserConfigurationException
     */
    protected String parseIncidentDeviceSysId(String soapResult) throws SAXException, IOException, ParserConfigurationException {

            String sysId = "";

            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();

            dbf.setNamespaceAware(false);
            DocumentBuilder docBuilder = dbf.newDocumentBuilder();

            InputSource inputSource = new InputSource();
            inputSource.setCharacterStream(new StringReader(soapResult));

            Document xmlDocument = docBuilder.parse(inputSource);

            Element elRoot = xmlDocument.getDocumentElement();

            NodeList listNodes = elRoot.getElementsByTagName("getRecordsResult");

            
            if ((listNodes != null) && (listNodes.getLength() > 0)) {
            
                int nodeLen = listNodes.getLength();
                int count =0;
                
                while(count < nodeLen)
                {
                    Element elNode = (Element)listNodes.item(count);
                    
                    NodeList classNames = elNode.getElementsByTagName("sys_class_name");
                    
                    NodeList elSysId = elNode.getElementsByTagName(getConfigurationItemField());
                    
                    if (elSysId.getLength()> 0)
                    {
                        sysId = elSysId.item(0).getTextContent();
                        
                        if (classNames.getLength()>0)
                        {
                            String className =   classNames.item(0).getTextContent();
                            
                            if (!"cmdb_ci_network_adapter".equals(className))
                            {
                                break;
                            }
                        }
                    }
                    
                    count++;
                }
            }

            return sysId;
    }

    protected String getIncidentSOAPURI() {
        
        if(svcContext.getAttribute(ServiceContext.INCIDENT_ENDPOINT)!=null)
        {
            return svcContext.getAttribute(ServiceContext.INCIDENT_ENDPOINT);
        }
        
        return "/incident.do?SOAP";
    }


    protected String getIncidentFieldName() {

        return "number";
    }
    
    protected String getConfigurationItemField(){
        return "sys_id";
    }
   
}
