package com.accelops.phoenix.servicenow;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.codec.Charsets;
import org.apache.commons.lang.StringUtils;
import org.apache.http.entity.ContentType;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import com.accelops.service.ExternalObjectInfo;
import com.accelops.service.ServiceContext;
import com.accelops.service.exception.DeviceIntegratedException;
import com.accelops.service.exception.SOAPAuthorizationException;
import com.accelops.service.exception.ServiceNowIntegrationException;

/**
*  Abstract class
*
*/
abstract class AbstractServiceNowIntegration implements ServiceNowIntegration{

	
	static final Logger logger = Logger.getLogger(AbstractServiceNowIntegration.class.getName());
	
	final static String DEFAULT_XSL_PATH =  "/opt/phoenix/config/transform/ServiceNow";
	
    private static final String CONFIG_FILE = "/opt/phoenix/config/transform/ServiceNow/config.properties";
	
	protected String integratedXML;
	
	protected ServiceContext svcContext;
	
	protected String xslPath;
    
	private ServiceNowSoapClient soapClient;
	
	 // initialize external object list to return newly created objects
    
    protected List<ExternalObjectInfo> externalObjList;
    
    protected String serviceNowHost;
    
    protected Properties configProp;
    
	AbstractServiceNowIntegration(ServiceContext context){
		this.svcContext = context;
		
		externalObjList = new ArrayList<ExternalObjectInfo>();
	}
	
	public void integrating() throws Exception
	{
		integratedXML = (String) svcContext.getAttribute(ServiceContext.KEY_XML);

        String xslPath = getXslPath();
        
        if (StringUtils.isBlank(integratedXML)) {
            logger.log(Level.WARNING, "ServiceNow:  Invalid source XML.");
            svcContext.setAttribute(ServiceContext.KEY_OUTPUT, 0);
            return;
        }
        
        if (StringUtils.isBlank(xslPath)) {
            logger.log(Level.WARNING, "ServiceNow:  Invalid xsl path.");
            svcContext.setAttribute(ServiceContext.KEY_OUTPUT, 0);
            return;
        }
        
        String url = (String) svcContext.getAttribute(ServiceContext.KEY_URL);
        
        if (StringUtils.isBlank(url)) {
            logger.log(Level.WARNING, "ServiceNow:  Invalid URL.");
            svcContext.setAttribute(ServiceContext.KEY_OUTPUT, 0);
            return;
        }
        serviceNowHost = getHost(url);
        
        
        loadConfigPropties();
        
		try 
		{
			execute();
		}
		catch (Exception integExp) 
		{
			// Throw Integration Exception
			//
		    throw integExp;
		}
		
		// put external object info in context output parameters
        //
		svcContext.setOutputParameters((ExternalObjectInfo[])this.externalObjList.toArray(new ExternalObjectInfo[externalObjList.size()]));
	}
	
	
	protected void loadConfigPropties() {
	    
	    configProp = new Properties();
	    try{
	        
	        configProp.load(new FileInputStream(CONFIG_FILE));    
	    }
	    catch(Exception ex)
	    {
	        
	    }
    }
	
	

    protected String getIntegratedXML()
	{
		return integratedXML;
	}
	
	
	protected String getXslPath()
	{
		
		 // If 
		 // 
		 if (xslPath == null)
	     {
	        xslPath = DEFAULT_XSL_PATH;
	     }
		 
		 return xslPath;
	}
	
	 private String getHost(String url){
	        
    	 if (url.toLowerCase().startsWith("http"))
         {
             int index = url.indexOf("://");
             
             if (index!=-1)
             {
                 
                url  = url.substring(index+3);
             }
         }
     
         if (url.indexOf("/") > 0)
         {
             url = url.substring(0, url.indexOf("/"));
         }
        
         return url;
    }
	 
	 
	 /**
     * 
     * @param soapResult
     * @return sys_id -- id from service now
     * 
     * @throws SAXException
     * @throws IOException
     * @throws ParserConfigurationException
     */
	 protected String parseSoapResultSysId(String soapResult) throws SAXException, IOException, ParserConfigurationException 
	 {

            String sysId = "";

            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();

            dbf.setNamespaceAware(false);
            DocumentBuilder docBuilder = dbf.newDocumentBuilder();

            InputSource inputSource = new InputSource();
            inputSource.setCharacterStream(new StringReader(soapResult));

            Document xmlDocument = docBuilder.parse(inputSource);

            Element elRoot = xmlDocument.getDocumentElement();

            NodeList listSysId = elRoot.getElementsByTagName("sys_id");

            if ((listSysId != null) && (listSysId.getLength() > 0)) {
                
                Node elSysId = listSysId.item(0);
                
                sysId = elSysId.getTextContent();
            }

            return sysId;
	 }
	 
	 
	 protected void faultProcess(String statusCode)
	 {
	        if (String.valueOf(statusCode).startsWith("4") )
	        {
	            throw new SOAPAuthorizationException("no permission to access ServiceNow");
	        }
	        else if (String.valueOf(statusCode).startsWith("5") && StringUtils.isNotBlank(soapClient.getSoapResultMessage()))
	        {
	            throw new ServiceNowIntegrationException(parseSoapFault(soapClient.getSoapResultMessage()));
	        }
	        else if (StringUtils.isNotBlank(soapClient.getSoapResultMessage()))
	        {
	            throw new ServiceNowIntegrationException(soapClient.getSoapResultMessage());
	        }
	        
	        throw new ServiceNowIntegrationException("Server Down or other issues, code: " + statusCode);
	 }
	    
	 protected void faultProcess(String custId,String hostIp, String hostName, String statusCode) 
	 {
	        if (String.valueOf(statusCode).startsWith("4") )
	        {
	            throw new SOAPAuthorizationException("no permission to access ServiceNow");
	        }
	        else if (StringUtils.isNotBlank(soapClient.getSoapResultMessage()))
	        {
	            throw new DeviceIntegratedException(parseSoapFault(soapClient.getSoapResultMessage()),custId,hostIp,hostName);
	        }
	        throw new DeviceIntegratedException("Unknown Issue",custId,hostIp,hostName);
	        
	 }
	    
	    /**
	     * 
	     * @param soapResult
	     * @return
	     */
	 protected String parseSoapFault(String soapResult) 
	 {

	        String faultString = "";
	        try
	        {
	            Document xmlDocument = XMLUtils.createDocument(soapResult, false);
	            
	            Element elRoot = xmlDocument.getDocumentElement();
	            
	            NodeList listSysId = elRoot.getElementsByTagName("faultstring");
	    
	            if ((listSysId != null) && (listSysId.getLength() > 0)) {
	                Element elSysId = (Element)listSysId.item(0);
	                faultString = elSysId.getFirstChild().getTextContent();
	            }
	        }
	        catch(Exception ex){
	            faultString = soapResult;
	        }
	        
	        return faultString;        
	 }
	 
	 /**
     * 
     * @param sourceXml
     * @param elementName
     * @return
     * @throws ParserConfigurationException
     * @throws SAXException
     * @throws IOException
     */
    protected String getElementValue(String sourceXml, String elementName) throws ParserConfigurationException, SAXException, IOException {
        
            String elementValue = "";

            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();

            dbf.setNamespaceAware(false);
            DocumentBuilder docBuilder = dbf.newDocumentBuilder();

            InputSource inputSource = new InputSource();
            inputSource.setCharacterStream(new StringReader(sourceXml));

            Document xmlDocument = docBuilder.parse(inputSource);

            Element elRoot = xmlDocument.getDocumentElement();

            NodeList listExtId = elRoot.getElementsByTagName(elementName);

            if ((listExtId != null) && (listExtId.getLength() > 0)) {
                    Element elExtId = (Element)listExtId.item(0);
                    elementValue = elExtId.getFirstChild().getNodeValue();
            }

            return elementValue;     
    }
	    
	 protected ServiceNowSoapClient getSoapClient() {
	     
	     if (soapClient == null)
	     {
	         soapClient = createSOAPClient();
	     }
        return soapClient;
    }
	 
	 
	protected ServiceNowSoapClient createSOAPClient()
	{
	     String user = (String) svcContext.getAttribute(ServiceContext.KEY_USER);
         String pwd = (String) svcContext.getAttribute(ServiceContext.KEY_PWD);
         
         ContentType contentType = Default_ContentType;
         
         if (!configProp.isEmpty() && configProp.containsKey("Content-Type"))
         {
             contentType = ContentType.create(configProp.getProperty("Content-Type"),Charsets.UTF_8);
         }
         
         ServiceNowSoapClient soapClient = new ServiceNowSoapClient(serviceNowHost,443,user,pwd,contentType);
         return soapClient;
	}

    abstract protected void execute() throws Exception;

    protected String getIncidentFieldName() {
        return "number";
    }
}
